/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#ifndef CONFIG_HANDLER_H
#define CONFIG_HANDLER_H

#include <QString>
#include <QMap>
#include <QCoreApplication>

namespace Ui {
class ConfigHandler;
}

class ConfigHandler
{

public:
	ConfigHandler(QString filename);
	QMap	<QString, QString> configmap;	//Store the config values
	QString	GetConfigValue(QString val);

	//UART
	int get_channel_count();
	QString get_channel_select();
	QString	get_output_port();
	QString get_SGX_port(int intrasys_num);

	//RCM
	int get_RCM_port_TCP();
	QString	get_RCM_port_USB();
	QString	get_RCM_port_RS232();
	QString get_RCM_port_RS485();
	bool get_support_RCM_mode_USB_live();
	bool get_support_RCM_mode_USB_blind();
	bool get_support_RCM_mode_RS232_live();
	bool get_support_RCM_mode_RS232_blind();
//	bool get_support_RCM_mode_RS485_live();
	bool get_support_RCM_mode_RS485_blind();
	bool get_support_RCM_mode_TCP_live();
	bool get_support_RCM_mode_TCP_blind();

	//SYSTEM
	bool	get_cursor_enabled();
	bool	get_fullscreen_enabled();
	int		get_polling_rate_data();
	bool	get_continuous_readings_enabled();
	bool	get_console_output_enabled();
	bool	get_reset_on_startup();
	bool	get_init_failed_exit();
	bool	get_support_DVGA();
	int		get_read_temperature();
	double	get_read_temperature_SG();
	bool	get_read_power();
	bool	get_duty_cycle_compensation_enabled();
	int		get_error_clear_mode();				//0 = manual clearing / 1 auto clearing, except RCM/modbus/etc.
	int		get_ok_signal_pin();
	bool	get_ODC_workaround_enabled();
	QString	get_rotary_switch_pins();
	bool	get_support_HAT_B14_0835();
	bool	get_enforce_software_limits_frequency();
	bool	get_enforce_software_limits_power();
	bool	get_enforce_software_limits_autocorrect();
	bool	get_use_runtime_based_reset_detection();
	bool	get_report_PA_type();
	//
	// TODO:
	// Home menu power control management
	// These should only be temporary and can be deleted later when an automated solution is provided.
	//
	int		get_feedforward_PWRS_controls_enabled();
	int		get_feedforward_mode();
	bool	get_support_negative_attenuation();

	//GPIOs
	int		get_GPIO_3v3_out();
	int		get_GPIO_interlock_in();

	//MENUS AND FEATURES
	bool	get_menu_home_enabled();
	bool	get_menu_ALL_enabled();
	bool	get_menu_DLL_enabled();
	bool	get_menu_sweep_enabled();
	bool	get_menu_settings_enabled();
	bool	get_menu_power_enabled();
	bool	get_menu_PSU_enabled();
	bool	get_menu_splitter_enabled();
	bool	get_menu_help_enabled();
	bool	get_menu_PID_enabled();
	bool	get_support_PWM_triggering();
	bool	get_support_external_triggering_mode();
	int		get_external_triggering_synchronization_enable();
	int		get_external_triggering_synchronization_delay_us();
	bool	get_support_coherency_standalone();
	bool	get_support_coherency_LVDS_master();
	bool	get_support_coherency_LVDS_slave_end();
	bool	get_support_coherency_LVDS_slave_inline();
	bool	get_support_coherency_AUX_master();
	bool	get_support_coherency_AUX_slave_end();
	bool	get_support_AIS_mode();
	bool	get_support_modbus_mode();
	bool	get_support_ethercat_mode();
	bool	get_support_logging();
	bool	get_support_feedforward_control();
	int		get_support_amplifier_mode();
	bool	get_support_PID_control();
	bool	get_support_power_control_modes();
    bool    get_support_phase_control();
	bool	get_support_power_offset();
	bool	get_controls_PWM_enabled();
	bool	get_controls_CW_enabled();
	bool	get_controls_ALL_enabled();
	bool	get_controls_DLL_enabled();

	//DEFAULTS
	bool	get_target_RF_enable();
	int		get_target_PA_type();
	int		get_remote_command_mode();
	int		get_coherency_mode();
	int		get_ext_triggering_mode();
	int		get_PWM_triggering_mode();
	bool	get_default_DVGA_amplifier_forward_enabled();
	bool	get_default_DVGA_amplifier_reflected_enabled();
	double	get_default_DVGA_attenuation_forward();
	double	get_default_DVGA_attenuation_reflected();
	double	get_PWM_minimum_pulse_length_ns();
	bool	get_use_minimum_pulse_length();
	bool	get_modbus_enabled();
	bool	get_ethercat_enabled();
	int		get_logging_rate();
	QString get_logging_directory();
	bool	get_logging_enabled();
	double	get_FFWD_attenuation_override();
	int		get_power_control_mode();			//0 = Normal, 1 = Analog Input Mode, 2 = Feed Forward, 3 = PID

	//CW MODE
	double	get_frequency_min();
	double	get_frequency_max();
	double	get_frequency_value();
	double	get_power_watt_min();
	double	get_power_watt_max();
	double	get_power_watt_value();
	double	get_power_dbm_value();
	double	get_phase_value();

	//PWM MODE
	bool	get_PWM_enabled();
	double	get_PWM_duty_cycle_value();
	double	get_PWM_frequency_value();
	int		get_PWM_frequency_min();
	int		get_PWM_frequency_max();
	int		get_PWM_slave_port();			//I don't know what this does
	int		get_PWM_slave_pin();			//I don't know what this does
	int		get_PWM_master_port();			//I don't know what this does
	int		get_PWM_master_pin();			//I don't know what this does
	double	get_PWM_refresh_rate();			//I don't know what this does	//returned by DCG, but never set?
	double	get_PWM_trigger_pulse();		//I don't know what this does	//returned by DCG, but never set?

	//ALL MODE
	bool	get_ALL_enabled();
	double	get_ALL_frequency_min();		//Voltage input limit
	double	get_ALL_frequency_max();		//Voltage input limit
	double	get_ALL_freqLimit_UP_value();
	double	get_ALL_freqLimit_DOWN_value();
	double	get_ALL_threshold_value();
	double	get_DVGA_forward_attenuation();
	double	get_DVGA_reflected_attenuation();
	bool	get_DVGA_forward_amplifier_enabled();
	bool	get_DVGA_reflected_amplifier_enabled();

	//DLL MODE
	bool	get_DLL_enabled();
	double	get_DLL_freqLimit_UP();
	double	get_DLL_freqLimit_DOWN();
	double	get_DLL_freqLimit_step();
	double	get_DLL_frequency_step();
	double	get_DLL_frequency_start();
	double	get_DLL_threshold_db();
	double	get_DLL_delay_ms();

	//SWEEP MODE
	double	get_SWP_start_freq();
	double	get_SWP_stop_freq();
	double	get_SWP_step_freq();
	double	get_SWP_power_watt();
	double	get_SWP_power_dbm();
	double	get_SWP_maximum_measure_points();

	//PSU
	int		get_PSU_count();
	bool	get_support_PSU_controls_enable();
	bool	get_support_PSU_controls_voltage();
	bool	get_support_PSU_controls_current_limit();
	double	get_PSU_voltage();
	double	get_PSU_current_limit();
	int		get_PSU_enabled();

	//NCHANNEL
	bool	get_splitter_channel_disable();
	int		get_splitter_channel_count();
	bool	get_support_splitter_controls_phase();
	bool	get_support_splitter_controls_magnitude();
	bool	get_support_splitter_controls_attenuation();

	//ANALOG INPUT MODE
	int		get_AIS_ADC_max();
	int		get_AIS_ADC_min();
	double	get_AIS_attenuation_value();    //static GCS value for the IQ modulator
	double	get_AIS_attenuation_max_db();
	double	get_AIS_attenuation_min_db();
	double	get_AIS_magnitude_max();
	double	get_AIS_magnitude_min();

	//AMPLIFIER MODE
	bool get_support_AM_phase_control();
	double	get_AM_attenuation_max_db();
	double	get_AM_attenuation_min_db();
	int get_AM_controls_enabled();

private:
	void	ReadConfig(QString filename);

	/* Serial Communication */
	QString SGX_port = "";
	QString	output_port = "";
	QString channel_select = "1";
	int channel_count = 1;

	/* Remote Command Mode */
	int	RCM_port_TCP = 9001;
	QString RCM_port_USB = "";
	QString RCM_port_RS232 = "";
	QString RCM_port_RS485 = "";

	/* System functionality enables */
	bool    cursor_enabled = true;
	bool	fullscreen_enabled = true;
	bool	continuous_readings_enabled = true;
	bool	console_output_enabled = false;
	bool	support_DVGA = false;
	bool	reset_on_startup = true;
	bool	init_failed_exit = false;
	int		ok_signal_pin = -1;
	bool	ODC_workaround_enabled = false;
	int		read_temperature = 0;		//TODO: Automate this
	double	read_temperature_SG = 0;
	bool	read_power = true;
	bool	duty_cycle_compensation_enabled = false;
	int		error_clear_mode = 0;
	QString	rotary_switch_pins = "";
	bool	splitter_channel_disable = 0;
	int		splitter_channel_count = 0;
	bool	support_splitter_controls_phase = false;
	bool	support_splitter_controls_magnitude = false;
	bool	support_splitter_controls_attenuation = false;
	bool	support_HAT_B14_0835 = false;
	bool	enforce_software_limits_frequency = true;
	bool	enforce_software_limits_power = true;
	bool	enforce_software_limits_autocorrect = true;
	bool	use_runtime_based_reset_detection = false;
	bool	report_PA_type = false;
	//
	// TODO:
	// Home menu power control management
	// These should only be temporary and can be deleted later when an automated solution is provided.
	//
	int		feedforward_PWRS_controls_enabled = -1 ;
	int		feedforward_mode = -1;
	bool	support_negative_attenuation = false;

	// GPIOs
	int		GPIO_3v3_out = -1;
	int		GPIO_interlock_in = -1;

	/* GUI menu and feature enables */
	bool	menu_power_enabled = false;
	bool	menu_settings_enabled = true;
	bool	menu_sweep_enabled = true;
	bool	menu_ALL_enabled = true;
	bool	menu_DLL_enabled = true;
	bool	menu_home_enabled = true;
	bool	menu_PID_enabled = true;
	bool	menu_PSU_enabled = true;
	bool	menu_splitter_enabled = false;
	bool	menu_help_enabled = true;
	bool	support_PWM_triggering = false;
	bool	support_external_triggering_mode = false;
	int		external_triggering_synchronization_enable = -1;
	int		external_triggering_synchronization_delay_us = -1;
	bool	support_AIS_mode = false;
	bool	support_coherency_standalone = false;
	bool	support_coherency_LVDS_master = false;
	bool	support_coherency_LVDS_slave_end = false;
	bool	support_coherency_LVDS_slave_inline = false;
	bool	support_coherency_AUX_master = false;
	bool	support_coherency_AUX_slave_end = false;
	bool	support_modbus_mode = false;
	bool	support_ethercat_mode = false;
	bool	support_logging = true;
	bool	support_feedforward_control = false;
	int		support_amplifier_mode = 0;
	bool	support_PID_control = false;
	bool	support_power_control_modes = true;
	bool    support_phase_control = true;
	bool	support_power_offset = false;
	bool	controls_CW_enabled = true;
	bool	controls_PWM_enabled = true;
	bool	controls_ALL_enabled = true;
	bool	controls_DLL_enabled = true;

	/* Remote Command Mode */
	bool	support_RCM_mode_USB_live = true;
	bool	support_RCM_mode_USB_blind = true;
	bool	support_RCM_mode_RS232_live = false;
	bool	support_RCM_mode_RS232_blind = false;
//	bool	support_RCM_mode_RS485_live = false;
	bool	support_RCM_mode_RS485_blind = false;
	bool	support_RCM_mode_TCP_live = true;
	bool	support_RCM_mode_TCP_blind = true;

	/* Important defaults */
	int     polling_rate_data = 200;
	int		target_PA_type = -1;
	bool	target_RF_enable = false;
	bool	default_DVGA_amplifier_forward_enabled = false;
	bool	default_DVGA_amplifier_reflected_enabled = false;
	double	default_DVGA_attenuation_forward = 9;
	double	default_DVGA_attenuation_reflected = 9;
	double	PWM_minimum_pulse_length_ns = 50000;
	bool	use_minimum_pulse_length = true;
	int		target_remote_command_mode = 0;
	int 	target_ext_trig_mode = 0;			//0 = OFF, 1 = Enabled -> PWM DC=0%
	int		target_clock_source = 0;			//0 = standalone mode, 1 = LVDS Master, 2 = LVDS Slave End, 3 = LVDS Slave Inline, 4 = AUX Master, 5 = AUX Slave End
	int		target_PWM_trigger_mode;			//1 = Freerunning, 2 = Triggered master, 3 = Triggered Slave
	bool	modbus_enabled = false;
	bool	ethercat_enabled = false;
	QString logging_directory = QCoreApplication::applicationDirPath() + "/logging";
	int		logging_rate = 1000;
	bool	logging_enabled = false;
	double	FFWD_attenuation_override = -1;

	/* CW mode */
	double	power_watt_min = -1;			//Power limit (Min)
	double	power_watt_max = -1;			//Power limit (Max)
	double	power_watt_value = 0.1;			//Starting value Power	(Watt)
	double	power_dbm_value = 20.0;			//Starting value Power (dBm) <- Not assigned from config
	double	frequency_min = 2400000000.0;	//Frequency limit (min)
	double	frequency_max = 2500000000.0;	//Frequency limit (max)
	double	freq_value = 2450000000.0;		//Starting value Frequency
	double	phase_value = 0.0;				//Starting value Phase

	/* ALL mode */
	bool	target_ALL_enabled = false;
	double	ALL_freqLimit_UP_value = 1.2;
	double	ALL_freqLimit_DOWN_value = 0.85;
	double	ALL_frequency_min = 0;
	double	ALL_frequency_max = 1.5;
	double	ALL_threshold_value = 0.2;

	/* DVGAs - Digital (Controlled) Variable Gain Amplifiers/Attenuators */
	double	DVGA_attenuation_forward = 9.0;
	double	DVGA_attenuation_reflected = 9.0;
	bool	DVGA_amplifier_forward_enabled = false;
	bool	DVGA_amplifier_reflected_enabled = false;

	/* DLL mode */
	bool	target_DLL_enabled = false;
	double	DLL_freqLimit_DOWN_value = 2400.0;
	double	DLL_freqLimit_UP_value = 2500.0;
	double	DLL_freqLimit_step = 0.1;
	double  DLL_freq_start_value = 2400.0;
	double  DLL_freq_step_value = 1.0;
	double	DLL_threshold_value = 0.5;
	double  DLL_main_delay_value = 0;

	/* PWM mode */
	bool	target_PWM_enabled = false;
	double	PWM_duty_cycle_value = 100.0;
	double 	PWM_freq_value = 1000;
	double	PWM_freq_min = 1000;
	double	PWM_freq_max = 19800;
	int		PWM_refresh_rate = 0;		//Purpose Unknown
	int		PWM_trigger_pulse = 0;		//Purpose Unknown
	int		PWM_master_port = 0;		//Purpose Unknown
	int		PWM_master_pin = 0;			//Purpose Unknown
	int		PWM_slave_port = 0;			//Purpose Unknown
	int		PWM_slave_pin = 0;			//Purpose Unknown

	/* Power Control Mode */
	int power_control_mode = 0;			//0 = Normal, 1 = Analog Input Mode, 2 = Feed Forward

	/* Analog Input Mode */
	double	AIS_attenuation_value;		//static VGA attenuation when modulating with IQ-mod.
	int		AIS_ADC_min;
	int		AIS_ADC_max;
	double	AIS_attenuation_max_db;
	double	AIS_attenuation_min_db;
	double	AIS_magnitude_max;
	double	AIS_magnitude_min;

	/* Amplifier Mode */
	bool support_AM_phase_control = false;
	double	AM_attenuation_max_db = 31.5;
	double	AM_attenuation_min_db = 0;
	int AM_controls_enabled = -1;

	/* Sweeping */
	double		SWP_start_freq = 2400000000.0;
	double		SWP_stop_freq = 2500000000.0;
	double		SWP_step_freq = 5000000.0;
	double		SWP_maximum_measure_points = 51;
	double		SWP_power_watt = 0.001;		//relies on dbm value above		//cannot be 0!
	double		SWP_power_dbm = 0.0;

	/* PSU */
	int		PSU_count = 0;
	double	PSU_voltage = -1;
	double	PSU_current_limit = -1;
	int		PSU_enabled = -1;
	bool	support_PSU_controls_enable = false;
	bool	support_PSU_controls_voltage = false;
	bool	support_PSU_controls_current_limit = false;

};

#endif // CONFIG_HANDLER_H
